@extends('layouts.admin')

@section('title', 'Error Details')

@section('content')
<div class="dashboard-header">
    <div class="d-flex justify-content-between align-items-center">
        <div>
            <h1>Error Details #{{ $error->id }}</h1>
            <p>
                <span class="status-badge status-{{ $error->level_color }}">{{ strtoupper($error->level) }}</span>
                @if($error->is_resolved)
                    <span class="status-badge status-completed ml-2">Resolved</span>
                @else
                    <span class="status-badge status-pending ml-2">Unresolved</span>
                @endif
                @if($error->is_sent)
                    <span class="status-badge status-info ml-2">
                        <i class="fas fa-paper-plane"></i> Sent to Support
                    </span>
                @endif
            </p>
        </div>
        <div>
            <a href="{{ route('admin.errors.index') }}" class="btn btn-secondary">
                <i class="fas fa-arrow-left"></i> Back to List
            </a>
        </div>
    </div>
</div>

<div class="row">
    <div class="col-md-8">
        {{-- Error Information --}}
        <div class="dashboard-card mb-4">
            <div class="dashboard-card-header">
                <h3>Error Information</h3>
            </div>
            <div class="dashboard-card-body">
                <div class="form-group">
                    <label><strong>Type:</strong></label>
                    <code>{{ $error->type }}</code>
                </div>

                <div class="form-group">
                    <label><strong>Message:</strong></label>
                    <div class="alert alert-danger">
                        {{ $error->message }}
                    </div>
                </div>

                <div class="form-group">
                    <label><strong>File Location:</strong></label>
                    <code>{{ $error->file }}</code>
                    @if($error->line)
                        <span class="ml-2">Line: <strong>{{ $error->line }}</strong></span>
                    @endif
                </div>

                <div class="form-group">
                    <label><strong>URL:</strong></label>
                    <div>
                        <span class="badge bg-primary">{{ $error->method }}</span>
                        <code>{{ $error->url }}</code>
                    </div>
                </div>

                <div class="form-group">
                    <label><strong>IP Address:</strong></label>
                    <span>{{ $error->ip ?? 'Unknown' }}</span>
                </div>

                <div class="form-group">
                    <label><strong>User Agent:</strong></label>
                    <small>{{ $error->user_agent ?? 'Unknown' }}</small>
                </div>

                <div class="form-group">
                    <label><strong>Occurred:</strong></label>
                    <span>{{ $error->created_at->format('F d, Y \a\t H:i:s') }}</span>
                    <small class="text-muted">({{ $error->created_at->diffForHumans() }})</small>
                </div>
            </div>
        </div>

        {{-- Stack Trace --}}
        <div class="dashboard-card mb-4">
            <div class="dashboard-card-header">
                <h3>Stack Trace</h3>
            </div>
            <div class="dashboard-card-body">
                <pre style="max-height: 400px; overflow-y: auto; background: #1e1e1e; color: #d4d4d4; padding: 1rem; border-radius: 4px; font-size: 0.85rem;">{{ $error->trace }}</pre>
            </div>
        </div>

        {{-- Context Data --}}
        @if($error->context)
        <div class="dashboard-card mb-4">
            <div class="dashboard-card-header">
                <h3>Context Data</h3>
            </div>
            <div class="dashboard-card-body">
                <pre style="max-height: 300px; overflow-y: auto; background: #1e1e1e; color: #d4d4d4; padding: 1rem; border-radius: 4px; font-size: 0.85rem;">{{ json_encode($error->context, JSON_PRETTY_PRINT) }}</pre>
            </div>
        </div>
        @endif
    </div>

    <div class="col-md-4">
        {{-- Quick Actions --}}
        <div class="dashboard-card mb-4">
            <div class="dashboard-card-header">
                <h3>Actions</h3>
            </div>
            <div class="dashboard-card-body">
                <div class="quick-actions">
                    @if(!$error->is_resolved)
                    <button type="button" class="quick-action-btn" onclick="resolveError()">
                        <i class="fas fa-check"></i>
                        <span>Mark as Resolved</span>
                    </button>
                    @endif

                    @if(!$error->is_sent)
                    <button type="button" class="quick-action-btn" onclick="sendToSupport()">
                        <i class="fas fa-paper-plane"></i>
                        <span>Send to Support</span>
                    </button>
                    @endif

                    <a href="{{ route('admin.errors.download', $error) }}" class="quick-action-btn">
                        <i class="fas fa-download"></i>
                        <span>Download JSON</span>
                    </a>

                    <button type="button" class="quick-action-btn text-danger" onclick="deleteError()">
                        <i class="fas fa-trash"></i>
                        <span>Delete Error</span>
                    </button>
                </div>
            </div>
        </div>

        {{-- User Information --}}
        @if($error->user)
        <div class="dashboard-card mb-4">
            <div class="dashboard-card-header">
                <h3>User Information</h3>
            </div>
            <div class="dashboard-card-body">
                <div class="mb-3">
                    <img src="{{ $error->user->avatar }}" alt="{{ $error->user->name }}" style="width: 50px; height: 50px; border-radius: 50%;">
                </div>
                <div class="form-group">
                    <label><strong>Name:</strong></label>
                    <div>{{ $error->user->name }}</div>
                </div>
                <div class="form-group">
                    <label><strong>Email:</strong></label>
                    <div>{{ $error->user->email }}</div>
                </div>
                <div class="form-group">
                    <label><strong>ID:</strong></label>
                    <div>{{ $error->user->id }}</div>
                </div>
                <a href="{{ route('admin.users.show', $error->user) }}" class="btn btn-sm btn-primary w-100">
                    <i class="fas fa-user"></i> View User Profile
                </a>
            </div>
        </div>
        @else
        <div class="dashboard-card mb-4">
            <div class="dashboard-card-header">
                <h3>User Information</h3>
            </div>
            <div class="dashboard-card-body text-center">
                <i class="fas fa-user-slash" style="font-size: 2rem; opacity: 0.5;"></i>
                <p class="mt-2 mb-0">Guest User</p>
            </div>
        </div>
        @endif

        {{-- Similar Errors --}}
        @if($similarErrors->count() > 0)
        <div class="dashboard-card">
            <div class="dashboard-card-header">
                <h3>Similar Errors ({{ $similarErrors->count() }})</h3>
            </div>
            <div class="dashboard-card-body">
                <div class="list-group">
                    @foreach($similarErrors as $similar)
                    <a href="{{ route('admin.errors.show', $similar) }}" class="list-group-item list-group-item-action" style="background: rgba(255,255,255,0.05); border: 1px solid rgba(255,255,255,0.1); margin-bottom: 0.5rem; border-radius: 4px; color: inherit; text-decoration: none; display: block; padding: 0.75rem;">
                        <div class="d-flex justify-content-between align-items-center">
                            <div style="flex: 1; min-width: 0;">
                                <div style="font-size: 0.9rem; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;">
                                    {{ $similar->short_message }}
                                </div>
                                <small class="text-muted">{{ $similar->created_at->diffForHumans() }}</small>
                            </div>
                            <span class="status-badge status-{{ $similar->level_color }} ml-2">
                                {{ strtoupper($similar->level) }}
                            </span>
                        </div>
                    </a>
                    @endforeach
                </div>
            </div>
        </div>
        @endif
    </div>
</div>

@endsection

@push('scripts')
<script>
function resolveError() {
    if (!confirm('Mark this error as resolved?')) return;

    fetch('{{ route("admin.errors.resolve", $error) }}', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        }
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            showNotification(data.message, 'success');
            setTimeout(() => location.reload(), 1000);
        } else {
            showNotification(data.message, 'error');
        }
    });
}

function sendToSupport() {
    if (!confirm('Send this error report to Siede Studios support dashboard?')) return;

    const btn = event.target.closest('.quick-action-btn');
    const originalHtml = btn.innerHTML;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i><span>Sending...</span>';
    btn.disabled = true;

    fetch('{{ route("admin.errors.send", $error) }}', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        }
    })
    .then(r => r.json())
    .then(data => {
        showNotification(data.message, data.success ? 'success' : 'error');
        if (data.success) {
            setTimeout(() => location.reload(), 1500);
        } else {
            btn.innerHTML = originalHtml;
            btn.disabled = false;
        }
    })
    .catch(error => {
        showNotification('Network error: ' + error.message, 'error');
        btn.innerHTML = originalHtml;
        btn.disabled = false;
    });
}

function deleteError() {
    if (!confirm('Are you sure you want to delete this error log? This action cannot be undone.')) return;

    fetch('{{ route("admin.errors.destroy", $error) }}', {
        method: 'DELETE',
        headers: {
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        }
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            showNotification(data.message, 'success');
            setTimeout(() => window.location.href = '{{ route("admin.errors.index") }}', 1000);
        } else {
            showNotification(data.message, 'error');
        }
    });
}

function showNotification(message, type) {
    if (window.adminPanel && window.adminPanel.showNotification) {
        window.adminPanel.showNotification(message, type);
    } else {
        alert(message);
    }
}
</script>
@endpush